import React from 'react';
import { BrowserRouter as Router, Routes, Route, Link, useLocation } from 'react-router-dom';
import { Building2, Users, Info, BookOpen, Phone, Menu, X, ChevronRight, Search, Send, MapPin, Mail, Phone as PhoneIcon } from 'lucide-react';
import CandidateForm from './components/CandidateForm';

// Components
const Navbar = () => {
  const [isOpen, setIsOpen] = React.useState(false);

  return (
    <nav className="bg-white shadow-md fixed w-full z-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between h-16">
          <div className="flex items-center">
            <Link to="/" className="flex items-center">
              <Building2 className="h-8 w-8 text-blue-600" />
              <span className="ml-2 text-xl font-bold text-gray-900">ARSE RH</span>
            </Link>
          </div>
          
          <div className="hidden md:flex items-center space-x-8">
            <Link to="/empresas" className="text-gray-700 hover:text-blue-600">Para Empresas</Link>
            <Link to="/candidatos" className="text-gray-700 hover:text-blue-600">Para Candidatos</Link>
            <Link to="/sobre" className="text-gray-700 hover:text-blue-600">Sobre</Link>
            <Link to="/blog" className="text-gray-700 hover:text-blue-600">Blog</Link>
            <Link to="/contato" className="text-gray-700 hover:text-blue-600">Contato</Link>
            <button className="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700">
              Área do Cliente
            </button>
          </div>

          <div className="md:hidden flex items-center">
            <button onClick={() => setIsOpen(!isOpen)} className="text-gray-700">
              {isOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
            </button>
          </div>
        </div>
      </div>

      {/* Mobile menu */}
      {isOpen && (
        <div className="md:hidden bg-white border-t">
          <div className="px-2 pt-2 pb-3 space-y-1">
            <Link to="/empresas" className="block px-3 py-2 text-gray-700 hover:bg-gray-50">Para Empresas</Link>
            <Link to="/candidatos" className="block px-3 py-2 text-gray-700 hover:bg-gray-50">Para Candidatos</Link>
            <Link to="/sobre" className="block px-3 py-2 text-gray-700 hover:bg-gray-50">Sobre</Link>
            <Link to="/blog" className="block px-3 py-2 text-gray-700 hover:bg-gray-50">Blog</Link>
            <Link to="/contato" className="block px-3 py-2 text-gray-700 hover:bg-gray-50">Contato</Link>
            <button className="w-full text-left px-3 py-2 bg-blue-600 text-white rounded-md">
              Área do Cliente
            </button>
          </div>
        </div>
      )}
    </nav>
  );
};

const Breadcrumb = () => {
  const location = useLocation();
  const paths = location.pathname.split('/').filter(path => path);
  
  return (
    <div className="bg-gray-50 py-2">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex items-center space-x-2 text-sm text-gray-600">
          <Link to="/" className="hover:text-blue-600">Home</Link>
          {paths.map((path, index) => (
            <React.Fragment key={path}>
              <ChevronRight className="h-4 w-4" />
              <Link 
                to={`/${paths.slice(0, index + 1).join('/')}`}
                className="capitalize hover:text-blue-600"
              >
                {path}
              </Link>
            </React.Fragment>
          ))}
        </div>
      </div>
    </div>
  );
};

const HomePage = () => (
  <div className="pt-16">
    {/* Hero Section */}
    <div className="bg-gradient-to-r from-blue-600 to-blue-800 text-white">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-24">
        <h1 className="text-4xl md:text-5xl font-bold mb-8">
          Conectando Talentos e Oportunidades
        </h1>
        <p className="text-xl mb-12 max-w-2xl">
          Soluções completas em recrutamento e seleção para empresas e profissionais
        </p>
        <div className="grid md:grid-cols-2 gap-8">
          <div className="bg-white/10 backdrop-blur-lg rounded-lg p-8 hover:bg-white/20 transition">
            <Building2 className="h-12 w-12 mb-4" />
            <h2 className="text-2xl font-bold mb-4">Para Empresas</h2>
            <p className="mb-6">Encontre os melhores talentos para sua empresa com nossa expertise em recrutamento.</p>
            <Link to="/empresas" className="inline-block bg-white text-blue-600 px-6 py-3 rounded-md font-semibold hover:bg-gray-100">
              Solicitar Proposta
            </Link>
          </div>
          <div className="bg-white/10 backdrop-blur-lg rounded-lg p-8 hover:bg-white/20 transition">
            <Users className="h-12 w-12 mb-4" />
            <h2 className="text-2xl font-bold mb-4">Para Candidatos</h2>
            <p className="mb-6">Impulsione sua carreira com as melhores oportunidades do mercado.</p>
            <Link to="/candidatos/cadastro" className="inline-block bg-white text-blue-600 px-6 py-3 rounded-md font-semibold hover:bg-gray-100">
              Cadastrar Currículo
            </Link>
          </div>
        </div>
      </div>
    </div>

    {/* Trusted By Section */}
    <div className="py-16 bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <h2 className="text-center text-2xl font-bold text-gray-900 mb-12">
          Empresas que confiam em nosso trabalho
        </h2>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-8 items-center opacity-50">
          {/* Replace with actual company logos */}
          {Array.from({ length: 4 }).map((_, i) => (
            <div key={i} className="h-12 bg-gray-300 rounded"></div>
          ))}
        </div>
      </div>
    </div>

    {/* Services Overview */}
    <div className="py-16">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <h2 className="text-3xl font-bold text-center mb-12">Nossos Serviços</h2>
        <div className="grid md:grid-cols-3 gap-8">
          {[
            {
              title: "Recrutamento e Seleção",
              description: "Processo completo de identificação e seleção dos melhores talentos."
            },
            {
              title: "Hunting",
              description: "Busca ativa de profissionais altamente qualificados para posições estratégicas."
            },
            {
              title: "Gestão de Talentos",
              description: "Consultoria especializada em desenvolvimento e retenção de talentos."
            }
          ].map((service, i) => (
            <div key={i} className="bg-white p-6 rounded-lg shadow-md">
              <h3 className="text-xl font-bold mb-4">{service.title}</h3>
              <p className="text-gray-600 mb-4">{service.description}</p>
              <Link to="/empresas" className="text-blue-600 font-semibold hover:text-blue-800">
                Saiba mais →
              </Link>
            </div>
          ))}
        </div>
      </div>
    </div>

    {/* Testimonials */}
    <div className="py-16 bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <h2 className="text-3xl font-bold text-center mb-12">O que dizem sobre nós</h2>
        <div className="grid md:grid-cols-3 gap-8">
          {[
            {
              name: "João Silva",
              role: "Diretor de RH",
              company: "Tech Solutions",
              text: "A ARSE RH transformou nosso processo de recrutamento, trazendo talentos excepcionais para nossa equipe."
            },
            {
              name: "Maria Santos",
              role: "Gerente de Projetos",
              company: "Inovação SA",
              text: "Profissionalismo e eficiência em cada etapa do processo. Altamente recomendado!"
            },
            {
              name: "Pedro Costa",
              role: "CEO",
              company: "StartUp Brasil",
              text: "Parceria que faz a diferença. Encontraram exatamente os profissionais que precisávamos."
            }
          ].map((testimonial, i) => (
            <div key={i} className="bg-white p-6 rounded-lg shadow-md">
              <p className="text-gray-600 mb-4">"{testimonial.text}"</p>
              <div>
                <p className="font-semibold">{testimonial.name}</p>
                <p className="text-sm text-gray-500">{testimonial.role}, {testimonial.company}</p>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  </div>
);

const Footer = () => (
  <footer className="bg-gray-900 text-white">
    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
      <div className="grid md:grid-cols-4 gap-8">
        <div>
          <div className="flex items-center mb-4">
            <Building2 className="h-8 w-8 text-blue-400" />
            <span className="ml-2 text-xl font-bold">ARSE RH</span>
          </div>
          <p className="text-gray-400">
            Conectando talentos e oportunidades desde 2010.
          </p>
        </div>
        <div>
          <h3 className="text-lg font-semibold mb-4">Links Rápidos</h3>
          <ul className="space-y-2">
            <li><Link to="/empresas" className="text-gray-400 hover:text-white">Para Empresas</Link></li>
            <li><Link to="/candidatos" className="text-gray-400 hover:text-white">Para Candidatos</Link></li>
            <li><Link to="/sobre" className="text-gray-400 hover:text-white">Sobre Nós</Link></li>
            <li><Link to="/blog" className="text-gray-400 hover:text-white">Blog</Link></li>
          </ul>
        </div>
        <div>
          <h3 className="text-lg font-semibold mb-4">Contato</h3>
          <ul className="space-y-2 text-gray-400">
            <li className="flex items-center">
              <MapPin className="h-5 w-5 mr-2" />
              São Paulo, SP
            </li>
            <li className="flex items-center">
              <Mail className="h-5 w-5 mr-2" />
              contato@arserh.com.br
            </li>
            <li className="flex items-center">
              <PhoneIcon className="h-5 w-5 mr-2" />
              (11) 9999-9999
            </li>
          </ul>
        </div>
        <div>
          <h3 className="text-lg font-semibold mb-4">Newsletter</h3>
          <p className="text-gray-400 mb-4">
            Receba novidades e oportunidades
          </p>
          <form className="flex">
            <input
              type="email"
              placeholder="Seu e-mail"
              className="flex-1 px-4 py-2 rounded-l-md text-gray-900"
            />
            <button
              type="submit"
              className="bg-blue-600 px-4 py-2 rounded-r-md hover:bg-blue-700"
            >
              <Send className="h-5 w-5" />
            </button>
          </form>
        </div>
      </div>
      <div className="border-t border-gray-800 mt-12 pt-8 text-center text-gray-400">
        <p>&copy; 2024 ARSE RH. Todos os direitos reservados.</p>
      </div>
    </div>
  </footer>
);

function App() {
  return (
    <Router>
      <div className="min-h-screen flex flex-col">
        <Navbar />
        <Breadcrumb />
        <main className="flex-grow">
          <Routes>
            <Route path="/" element={<HomePage />} />
            <Route path="/candidatos/cadastro" element={<CandidateForm />} />
            {/* Add other routes here */}
          </Routes>
        </main>
        <Footer />
      </div>
    </Router>
  );
}

export default App;